function [cfg, artifact] = ft_artifact_muscle(cfg, data)

% FT_ARTIFACT_MUSCLE scans data segments of interest for muscle artifacts.
%
% Use as
%   [cfg, artifact] = ft_artifact_muscle(cfg)
% with the configuration options
%   cfg.dataset    = string with the filename
% or
%   cfg.headerfile = string with the filename
%   cfg.datafile   = string with the filename
% and optionally
%   cfg.headerformat
%   cfg.dataformat
%
% Alternatively you can use it as
%   [cfg, artifact] = ft_artifact_muscle(cfg, data)
% where the input data is a structure as obtained from FT_PREPROCESSING.
%
% In both cases the configuration should also contain
%   cfg.trl        = structure that defines the data segments of interest, see FT_DEFINETRIAL
%   cfg.continuous = 'yes' or 'no' whether the file contains continuous data
%
% Prior to artifact detection, the data is preprocessed (again) with the following
% configuration parameters, which are optimal for identifying muscle artifacts.
%   cfg.artfctdef.muscle.bpfilter    = 'yes'
%   cfg.artfctdef.muscle.bpfreq      = [110 140]
%   cfg.artfctdef.muscle.bpfiltord   = 8
%   cfg.artfctdef.muscle.bpfilttype  = 'but'
%   cfg.artfctdef.muscle.hilbert     = 'yes'
%   cfg.artfctdef.muscle.boxcar      = 0.2
%
% Artifacts are identified by means of thresholding the z-transformed value
% of the preprocessed data.
%   cfg.artfctdef.muscle.channel     = Nx1 cell-array with selection of channels, see FT_CHANNELSELECTION for details
%   cfg.artfctdef.muscle.cutoff      = z-value at which to threshold (default = 4)
%   cfg.artfctdef.muscle.trlpadding  = number in seconds (default = 0.1)
%   cfg.artfctdef.muscle.fltpadding  = number in seconds (default = 0.1)
%   cfg.artfctdef.muscle.artpadding  = number in seconds (default = 0.1)
%
% The output argument "artifact" is a Nx2 matrix comparable to the "trl" matrix of
% FT_DEFINETRIAL. The first column of which specifying the beginsamples of an
% artifact period, the second column contains the endsamples of the artifactperiods.
%
% To facilitate data-handling and distributed computing, you can use
%   cfg.inputfile   =  ...
% to read the input data from a *.mat file on disk. This mat files should contain
% only a single variable named 'data', corresponding to the input structure.
%
% See also FT_REJECTARTIFACT, FT_ARTIFACT_CLIP, FT_ARTIFACT_ECG, FT_ARTIFACT_EOG,
% FT_ARTIFACT_JUMP, FT_ARTIFACT_MUSCLE, FT_ARTIFACT_THRESHOLD, FT_ARTIFACT_ZVALUE

% Copyright (C) 2003-2011, Jan-Mathijs Schoffelen & Robert Oostenveld
%
% This file is part of FieldTrip, see http://www.fieldtriptoolbox.org
% for the documentation and details.
%
%    FieldTrip is free software: you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation, either version 3 of the License, or
%    (at your option) any later version.
%
%    FieldTrip is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with FieldTrip. If not, see <http://www.gnu.org/licenses/>.
%
% $Id$

% these are used by the ft_preamble/ft_postamble function and scripts
ft_revision = '$Id$';
ft_nargin   = nargin;
ft_nargout  = nargout;

% do the general setup of the function
ft_defaults
ft_preamble init

% ft_preamble provenance is not needed because just a call to ft_artifact_zvalue
% ft_preamble loadvar data is not needed because ft_artifact_zvalue will do this

% the ft_abort variable is set to true or false in ft_preamble_init
if ft_abort
  return
end

% check if the input cfg is valid for this function
cfg = ft_checkconfig(cfg, 'renamed',    {'datatype', 'continuous'});
cfg = ft_checkconfig(cfg, 'renamedval', {'continuous', 'continuous', 'yes'});
cfg = ft_checkconfig(cfg, 'forbidden',  {'padding', 'pretim', 'psttim'});

% set default rejection parameters
cfg.artfctdef               = ft_getopt(cfg,                  'artfctdef', []);
cfg.artfctdef.muscle        = ft_getopt(cfg.artfctdef,        'muscle',    []);
cfg.artfctdef.muscle.method = ft_getopt(cfg.artfctdef.muscle, 'method',    'zvalue');

if isfield(cfg.artfctdef.muscle, 'artifact')
  fprintf('muscle artifact detection has already been done, retaining artifacts\n');
  artifact = cfg.artfctdef.muscle.artifact;
  return
end

if ~strcmp(cfg.artfctdef.muscle.method, 'zvalue')
  ft_error('muscle artifact detection only works with method=''zvalue''');
end

% for backward compatibility
cfg.artfctdef.muscle = ft_checkconfig(cfg.artfctdef.muscle, 'renamed', {'sgn',     'channel'});
cfg.artfctdef.muscle = ft_checkconfig(cfg.artfctdef.muscle, 'renamed', {'passbnd', 'bpfreq'});
cfg.artfctdef.muscle = ft_checkconfig(cfg.artfctdef.muscle, 'renamed', {'padding', 'trlpadding'});

% settings for preprocessing
cfg.artfctdef.muscle.bpfilter    = ft_getopt(cfg.artfctdef.muscle, 'bpfilter',   'yes');
cfg.artfctdef.muscle.bpfreq      = ft_getopt(cfg.artfctdef.muscle, 'bpfreq',     [110 140]);
cfg.artfctdef.muscle.bpfiltord   = ft_getopt(cfg.artfctdef.muscle, 'bpfiltor',   8);
cfg.artfctdef.muscle.bpfilttype  = ft_getopt(cfg.artfctdef.muscle, 'bpfilttype', 'but');
cfg.artfctdef.muscle.hilbert     = ft_getopt(cfg.artfctdef.muscle, 'hilbert',    'yes');
cfg.artfctdef.muscle.boxcar      = ft_getopt(cfg.artfctdef.muscle, 'boxcar',     0.2);

% settings for the zvalue subfunction
cfg.artfctdef.muscle.channel    = ft_getopt(cfg.artfctdef.muscle, 'channel',    'MEG');
cfg.artfctdef.muscle.cutoff     = ft_getopt(cfg.artfctdef.muscle, 'cutoff',     4);
cfg.artfctdef.muscle.trlpadding = ft_getopt(cfg.artfctdef.muscle, 'trlpadding', 0.1);
cfg.artfctdef.muscle.fltpadding = ft_getopt(cfg.artfctdef.muscle, 'fltpadding', 0.1);
cfg.artfctdef.muscle.artpadding = ft_getopt(cfg.artfctdef.muscle, 'artpadding', 0.1);

% construct a temporary configuration that can be passed onto FT_ARTIFACT_ZVALUE
tmpcfg                  = cfg;
tmpcfg.artfctdef.zvalue = cfg.artfctdef.muscle;
tmpcfg.artfctdef        = rmfield(tmpcfg.artfctdef, 'muscle');

% call the zvalue artifact detection function, where the data is either passed
% into the function by the user or read from file with cfg.inputfile
hasdata = exist('data', 'var');
if ~hasdata
  tmpcfg             = ft_checkconfig(tmpcfg, 'dataset2files', 'yes');
  tmpcfg             = ft_checkconfig(tmpcfg, 'required',      {'headerfile', 'datafile'});
  [tmpcfg, artifact] = ft_artifact_zvalue(tmpcfg);
else
  tmpcfg.artfctdef.zvalue.trlpadding = 0;
  tmpcfg.artfctdef.zvalue.fltpadding = 0;
  ft_warning('trlpadding and fltpadding are set to zero to avoid filter problems with NaN, see bug3193 for details');
  [tmpcfg, artifact] = ft_artifact_zvalue(tmpcfg, data);
end
cfg.artfctdef.muscle = tmpcfg.artfctdef.zvalue;
